/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2011-2018 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Application
    inverseTaskBVS

Description
    Solves inverse task for the heat conduction problem.

\*---------------------------------------------------------------------------*/

//#include <dlib/optimization.h>

#include <math.h>
#include <nlopt.h>

#include "OFstream.H"
#include "IOmanip.H"
#include "fvCFD.H"
#include "simpleControl.H"

//- For interpolation functionality
#include "interpolateXY.H"
#include "interpolation.H"


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //


//  NLOPT: Objective function 
#include "objFunNLOPT.H"


//  Main program
int main(int argc, char *argv[])
{
	#include "setRootCaseLists.H"

	#include "createTime.H"
	#include "createMesh.H"

    simpleControl simple(mesh);

    //  Create variables
	#include "createFields.H"

    //  Create optimization model fields
    #include "createOptModelFields.H"

    


	//Info<< nl << T.db().names() << nl << endl;
	//lookupObject<scalarField>("HTC");

	//  Initial value for the DLIB optimization parameter
	//column_vector htcSlope = {0.};  //  Initial "slope"


    fileName outputFile(runTime.path()/"results.csv");
    OFstream os(outputFile);
    os << "Time [s],HTC,Texp,Tsim" << nl;
   

    //  Initial value for the NLOPT optimization parameter
    double minHTC = 0.;
    double maxHTC = 100000.;
    double htcSlope[1] = {0.0};  // initial guess 
    // double iniStep[1];  // initial step 

    double lb[1] = {-1.}; // lower bound
    double ub[1] = {-1.}; // upper bound
    
    double htcSlopeTolAbs[1];
    double minf;
    double scaleSlope = 6.4e6;  // rho * cp, for unknown reason works the best!

    int major, minor, bugfix;
    nlopt_version(&major, &minor, &bugfix);
    Info << "NLOPT version: " << major << "."
    << minor << "." << bugfix << endl;

    nloptData *ptr = new nloptData
    (
        runTime,
	    T,
	    DT,
        rho,
        cp,
	    HTC,
	    TempProbe,
        mesh,
        simple,
	    stabLoops,
        interpolationDict,
        sensors,
        sensorsCellI,
        scaleSlope
    );
    nlopt_opt opt;

    opt = nlopt_create(NLOPT_LN_BOBYQA, 1);
    //opt = nlopt_create(NLOPT_LN_NELDERMEAD, 1); //slower than BOBYQA
    //opt = nlopt_create(NLOPT_LN_COBYLA, 1);   //slower than NELDERMEAD
    //opt = nlopt_create(NLOPT_LN_SBPLX, 1);   //slower than COBYLA
    //opt = nlopt_create(NLOPT_LN_PRAXIS, 1);  // not converging
    //opt = nlopt_create(NLOPT_LN_NEWUOA_BOUND, 1); // crashed
    

    nlopt_set_min_objective(opt, myfuncNLOPT, ptr);
    //nlopt_set_xtol_rel(opt, 1e-14); 
    //nlopt_set_ftol_rel(opt, 1e-2);
    //nlopt_set_maxeval(opt, 10);
    htcSlopeTolAbs[0] =2./(scaleSlope * runTime.deltaT().value());
    nlopt_set_xtol_abs(opt, htcSlopeTolAbs); 

    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //
    
	while (runTime.run())
	{
		// set lower bounds
        lb[0] = minHTC - HTC[2842];
        // lower bound using prediction of htcSlope range
        lb[0] = max(-1000.,lb[0]); 
        lb[0] /= scaleSlope * runTime.deltaT().value();
        nlopt_set_lower_bounds(opt, lb);
              

        // set upper bounds
        ub[0] = maxHTC - HTC[2842];
        // upper bound using prediction of htcSlope range
        ub[0] = min(1000.,ub[0]);
        ub[0] /= scaleSlope * runTime.deltaT().value();
        nlopt_set_upper_bounds(opt, ub);
                
        htcSlope[0] = 1e-4;
                
        // zones
        forAll(mesh.cellZones(), zoneI)
        {
        
        const labelList& cellsInZone = mesh.cellZones()[zoneI]; 
        
            forAll(cellsInZone,cellID)
            {
                const label cId = cellsInZone[cellID];

                DT[cId]
		      =
			    interpolateXY
			    (
				    T[cId],
			       *(TDataList[zoneI]),
			       *(kDataList[zoneI])
			    );


                rho[cId]
		      =
			    interpolateXY
			    (
				    T[cId],
			       *(rhoTDataList[zoneI]),
			       *(rhoDataList[zoneI])
			    );


                cp[cId]
		      =
			    interpolateXY
			    (
				    T[cId],
			       *(cpTDataList[zoneI]),
			       *(cpDataList[zoneI])
			    );
            }
        }



        // correct boundary conditions
        DT.correctBoundaryConditions();
        rho.correctBoundaryConditions();
        cp.correctBoundaryConditions();


        // NLOPT optimization
        nlopt_optimize(opt, htcSlope, &minf);


        //iniStepNLOPT = nlopt_get_initial_step(opt,htcSlope,iniStep);
        const int nEvalsNLOPT = nlopt_get_numevals(opt);
        Info << "number of optimizations: " << nEvalsNLOPT 
             << " ,timeIndex: " << runTime.timeIndex() 
             << " ,current time: " << runTime.time().value() << nl;
        


		//  Final Direct Task Loop
		runTime++;
        
		//  Adjust HTC
		HTC += scaleSlope * htcSlope[0] * runTime.deltaT().value();

 
        while (simple.correctNonOrthogonal())
        {

		    //  Heat conduction problem
		    fvScalarMatrix TEqn
	        (
        	    fvm::ddt(rho*cp, T) - fvm::laplacian(DT, T)
	        );

		    //  Solve Direct Task
		    TEqn.solve();
        }


		//  Save calculated fields
		//#include "write.H"

		//  Save fields
		//runTime.write();


        // interpolate temperature in the position of each thermocouple
        autoPtr<interpolation<scalar>> Tinterp 
                             = 
                               interpolation<scalar>::New(interpolationDict, T);
        Foam::List<scalar> sensorsT(0);
        forAll(sensors,sensorI)
        {
            scalar sensorT = Tinterp->interpolate(sensors[0], sensorsCellI[0]);
            sensorsT.append(sensorT);
        }

       
        // write in a file
        os << setprecision(10) << runTime.time().value()-runTime.deltaT().value() << ", " 
           << setprecision(10)  << HTC[2842] << ", " 
           << setprecision(10)  << TempProbe[runTime.timeIndex()] << ", " 
           << setprecision(10)  << sensorsT[0] << endl;
    }
		//  Output calculation performance at the end of the loop
		Info<< "OPT LOOP <<<<  CPUTIME = " << runTime.elapsedCpuTime() << " s"
		    << "  RUNTIME = " << runTime.elapsedClockTime() << " s"
		    << nl << nl;
    

    Info<< "OPT >>>> HAPPY END\n" << endl;
    nlopt_destroy(opt);
    delete ptr;
    return 0;
}


// ************************************************************************* //
