// structure used to pass OpenFOAM data to NLOPT
typedef struct nloptData_t
{
    Time & runTime_;
    volScalarField & T_;
    volScalarField & DT_;
    volScalarField & rho_;
    volScalarField & cp_;
	volScalarField & HTC_;
    scalarField & tempProbe_;
    polyMesh & mesh_;
    simpleControl & simple_;
    int ofIter_;
    dictionary & interpolationDict_;
    Foam::List<vector> & sensors_;
    Foam::List<label> & sensorsCellI_;
    double scaleSlope_;

    nloptData_t(
    Time &runTime, 
    volScalarField &T, 
    volScalarField &DT,
    volScalarField & rho,
    volScalarField & cp,
	volScalarField & HTC,
    scalarField & tempProbe,
    polyMesh & mesh,
    simpleControl & simple,
    int ofIter,
    dictionary & interpolationDict,
    Foam::List<vector> & sensors,
    Foam::List<label> & sensorsCellI,
    double scaleSlope
    ) 
    : 
    runTime_(runTime), 
    T_(T), DT_(DT),
    rho_(rho), cp_(cp),
    HTC_(HTC),
    tempProbe_(tempProbe),
    mesh_(mesh),
    simple_(simple),
    ofIter_(ofIter),
    interpolationDict_(interpolationDict),
    sensors_(sensors),
    sensorsCellI_(sensorsCellI),
    scaleSlope_(scaleSlope)
    {}

} nloptData;


// objective function
double myfuncNLOPT(unsigned n, const double *x, double *grad, void *my_func_data)
{
    auto ptr = static_cast<nloptData*>(my_func_data);

    double slope = x[0];
    double scaleSlope = ptr->scaleSlope_;
    Time &runTime = ptr->runTime_; 
    volScalarField & T = ptr->T_;
    volScalarField & DT = ptr->DT_;
    volScalarField & rho = ptr->rho_;
    volScalarField & cp = ptr->cp_;
    volScalarField & HTC = ptr->HTC_;
    scalarField & tempProbe = ptr->tempProbe_;
    simpleControl & simple = ptr->simple_;
    const int ofIter = ptr->ofIter_;
    dictionary & interpolationDict = ptr->interpolationDict_;
    Foam::List<vector> & sensors = ptr->sensors_;
    Foam::List<label> & sensorsCellI = ptr->sensorsCellI_;

    //  Store pre-loop time
    const label tIndex = runTime.timeIndex();
    const dimensionedScalar tValue = runTime.time();
    //  Store old time values
    const volScalarField Told = T;
    //  Store HTC
    const volScalarField HTCold = HTC;
    //  Objective function
    double Func = 0.;
    
    
    for (int i=0; i < ofIter; i++)
	{
		runTime++;
        //  Adjust HTC
		HTC += scaleSlope * slope * runTime.deltaT().value();
     

        while (simple.correctNonOrthogonal())
        {
            // build Ax-b=0
            fvScalarMatrix TEqn
	        (
        	    fvm::ddt(rho*cp, T) - fvm::laplacian(DT, T)
	        );

		    //  Solve Direct Task
		    TEqn.solve();
        }


        
        // interpolate temperature in the position of each thermocouple
        autoPtr<interpolation<scalar>> Tinterp 
                             = 
                               interpolation<scalar>::New(interpolationDict, T);
        Foam::List<scalar> sensorsT(0);
        forAll(sensors,sensorI)
        {
            scalar sensorT = Tinterp->interpolate(sensors[0], sensorsCellI[0]);
            sensorsT.append(sensorT);
        }

       
        //  Update objective function
        // only sensorsT[0] currently considered, although more sensors in
        // transportproperties dict. This is for a future extension to IHCP with
        // multiple sensors
        const double delta = (sensorsT[0] - tempProbe[tIndex + i]);
        Func += delta*delta;
    }

    //  Move back in time by ofIter_
	//Info<< "    Reset time and values" << endl;
    runTime.setTime(tValue, tIndex);
    T = Told;
    HTC = HTCold;

    //  Return the objective function
    return Func;
}

