# Artificial Compressibility with Riemann Solvers: Convergence on Unstructured Meshes

In the paper, we presented a solver for modelling incompressible variable density flows, based on the method of artificial compressibility rather than the PIMPLE algorithm.

The ``code`` folder contains the solver ``kelecyPletcherFoam``, an improved gradient limiter ``cellLimited<Michalak>``, and the three cases presented in the paper: ``damBreakTest``, ``damBreak``, and``hydrostatic``.

As mentioned in the paper, the solver is not at the point where it can be used for practical cases. However, we hope that its source code will be useful to other researchers investigating artificial compressibility.

## Getting Started

### Ready

The solver and limiter were developed for and tested with OpenFOAM v1906.

The solver uses the C++ package [Armadillo](http://arma.sourceforge.net/) for matrix multiplication, so make sure to have that installed. We used version 8.4. You can install using
```bash
sudo apt-get install libarmadillo8
```
or
```bash
sudo apt-get install libarmadillo-dev
```
or, for the most recent version, using the [official instructions](http://arma.sourceforge.net/download.html). 

Optionally, you may wish to install [R](https://www.r-project.org/) so you can use the post-processing script for the ``hydrostatic`` case. We used version 3.4.4.

### Compile

In the main ``code`` folder, the solver and limiter can be compiled with
```bash
./Allwmake
```
and cleaned with
```bash
./Allwclean
```

### Go

Each tutorial has its own folder in the ``tutorials`` folder. In each tutorial folder, the simulation can be run with 
```bash
./Allrun
```
cleaned with
```bash
./Allclean
```
and have its residuals plotted with
```bash
gnuplot residuals
```

The ``hydrostatic`` case also contains an R script called ``slice.R`` which plots results from the ``singleGraph`` function object.

## Advanced Options

If you want to change the default settings, you need to delve into the dictionaries in each tutorial folder.

### Setting Parameters in ``fvSolution``

The dictionary looks like
```cpp
kelecyPletcher
{
    maxPseudoIters      100000;

    rhoTol              0.0001;
    rhoUTol             0.01;
    pTol                0.01;
      
    nRungeKuttaSteps    1;
        
    beta                1100.0;

    deltaTauCoNum       0.48;
    deltaTCoNum         2.0;

    deltaTau            0.000001;
    deltaT              0.05;

    transient           yes;
    specialGradp        yes;
}
```
where the parameters are defined as follows:
 - ``maxPseudoIters`` is the maximum number of pseudo-time iterations before the solver stops trying to achieve convergence
 - ``rhoTol`` is the absolute convergence tolerance for *ρ*, ``rhoUTol`` for *ρ**u***, and ``pTol`` for *p*
 - ``nRungeKuttaSteps`` is the number of Runge-Kutta steps taken each pseudo-time step
 - ``beta`` is the artificial compressibility coefficient *β*
 - ``deltaTauCoNum`` is the Courant number for pseudo-time *τ*
 - ``deltaTCoNum`` is the Courant number for real-time *t* (``maxCo`` in ``controlDict`` does nothing)
 - ``deltaTau`` is the first pseudo-time step *Δτ*
 - ``deltaT`` is the first real-time step *Δt* (``deltaT`` in ``controlDict`` does nothing)
 - ``transient`` is ``no`` if you only want one real-time step (then it saves pseudo-time iterations) or ``yes`` if you want multiple real-time steps (then it saves real-time iterations)
 - ``specialGradp`` is ``no`` if you want to use standard gradient schemes to calculate ``grad(p)`` or ``yes`` if you want to use the well-balanced pressure gradient calculation derived from the momentum equation

### Specifying Limiters in ``fvSchemes``

Gradient schemes need to be specified for the MUSCL reconstruction of ``rho``, ``U`` and---if ``specialGradp`` is set to ``no`` in ``fvSolution``---``p``. For instance:
```cpp
gradSchemes
{
    grad(rho)   cellLimited<Michalak> 2.0 Gauss linear 1;
    grad(U)     cellLimited<Michalak> 2.0 Gauss linear 1;
}
```
where the number after ``cellLimited<Michalak>`` is *y<sub>t<sub>*.

### Loading Libraries in ``controlDict``

To use ``cellLimtied<Michalak>``, make sure to specify the custom library
```cpp
libs
(
    "libMichalak.so"
);
```
in ``controlDict``.

## Caution!

Currently, there are a few suboptimalities in the practical implementation of the solver.

First, the parameter *K* in ``cellLimited<Michalak>`` can only be changed in the code itself (line 108 of ``code/cellLimitedGrad/cellLimitedGrad.C``). Make sure to compile the code again if you change it.

Second, the solver only works for wall boundary conditions at the moment. If you want to develop this further, note that the boundary conditions specified in the dictionaries *are* used to calculate the gradient at boundary cells, and this is used (indirectly if ``specialGradp`` is ``yes``) to extrapolate *p* to the wall---but they *are not* used to calculate the boundary fluxes. Boundary fluxes are set directly with the assumption of a wall boundary condition.

## License
[GNU](http://www.gnu.org/licenses/)
