/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           |
     \\/     M anipulation  |
-------------------------------------------------------------------------------
                            | Copyright (C) 2021 Shannon Leakey,
                            |               Newcastle University
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Application
    kelecyPletcherFoam

Description
    Godunov-type scheme for variable-density artificial-compressibility
    
    Make sure to install Armadillo (http://arma.sourceforge.net/)

\*---------------------------------------------------------------------------*/

#include "fvCFD.H"
#include <armadillo>
#include "roeFlux.H"
#include "calculateFlux.H"
#include "createRotationMatrices.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

int main(int argc, char *argv[])
{
    #include "setRootCase.H"
    #include "createTime.H"
    #include "createMesh.H"
    #include "createFields.H"

    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

    Info<< "\nStarting time loop\n" << endl;
    
    // Real time stepping
    while(runTime.run())
    {
        Info<< "\nReal-time: value = " << runTime.value()
            << ", delta = " << deltaT.value()
            << endl;

        if(transient == 1)
        {
            runTime.setDeltaT(deltaT);
            runTime++;
        }

        // Save last real time
        rhoN = rho;
        rhoUN = rhoU;
        PN = P;

        // Pseudo time stepping
        for(int m = 0; m < maxPseudoIters; m++)
        {
            Info<< "\nPseudo-time: iteration " << m
                << ", min(delta) = " << gMin(localDeltaTau)
                << ", max(delta) = " << gMax(localDeltaTau)
                << endl;

            if(transient == 0)
            {
                runTime.setDeltaT(gMin(localDeltaTau));
                runTime++;
            }

            // Save last pseudo time
            rhoM = rho;
            rhoUM = rhoU;
            PM = P;

            // Source terms
            volScalarField rho_t = (rhoM - rhoN)/deltaT;
            volVectorField rhoU_t = (rhoUM - rhoUN)/deltaT;
            volVectorField rhog = rhoM*g;

            // Runge Kutta iterations
            for(int k = 0; k < nRungeKuttaSteps; k++)
            {
                Info<< "Runge-Kutta: iteration " << k << endl;

                // Runge Kutta coefficient
                scalar RungeKuttaCoeff = 1/(nRungeKuttaSteps - k);
                volScalarField factor = RungeKuttaCoeff*localDeltaTau/(1 + RungeKuttaCoeff*localDeltaTau/deltaT);

                // Calculate conservative fluxes with primitive variables
                #include "gradients.H"
                #include "convectiveFlux.H"

                // Explicit update equation
                Info<< "Updating conserved variables" << endl;
                rho = rhoM + factor*(-fvc::surfaceIntegrate(F0) - rho_t);
                rhoU = rhoUM + factor*(-fvc::surfaceIntegrate(F123) - rhoU_t + rhog);
                P = PM + factor*(-fvc::surfaceIntegrate(F4));

                // Update primitive variables
                U = rhoU/rho;
                p = P*betaDim;
                rho.correctBoundaryConditions();
                U.correctBoundaryConditions();
                p.correctBoundaryConditions();
            }

            // Check if converged
            scalarField rhoDiff = mag(rho - rhoM);
            scalarField rhoUDiff = mag(rhoU - rhoUM);
            scalarField pDiff = (P - PM)*betaDim;

            scalar maxRhoDiff = gMax(rhoDiff);
            scalar maxRhoUDiff = gMax(rhoUDiff);
            scalar maxPDiff = gMax(pDiff);

            bool rhoConverged = (maxRhoDiff < rhoTol);
            bool rhoUConverged = (maxRhoUDiff < rhoUTol);
            bool pConverged = (maxPDiff < pTol);

            Info<< "Residuals: rho = " << maxRhoDiff
                << ", rhoU = " << maxRhoUDiff
                << ", p = " << maxPDiff
                << endl;

            #include "setDeltaTau.H"

            if(transient == 0)
            {
                Info<< "\nWriting" << endl;
                runTime.write();
                runTime.printExecutionTime(Info);
            }

            if((m > 1) & ((rhoConverged & rhoUConverged & pConverged) | (m == (maxPseudoIters - 1))))
            {
                break;
            }
        }

        if(transient == 1)
        {
            #include "setDeltaT.H"

            Info<< "\nWriting" << endl;
            runTime.write();
            runTime.printExecutionTime(Info);
        }
        else
        {
            // force writing last pseudo time step
            rho.write();
            U.write();
            p.write();
            gradRho.write();
            gradU.write();
            gradp.write();
            break;
        }

    }

    Info<< "End\n" << endl;

    return 0;
}


// ************************************************************************* //
