/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::checkFieldLimits

Description
    Computes the min, mean and max of a field and provides a warning, if 
    certain limites are exceeded. Helpful to find the first problematic 
    equation / field for a diverging simulation - because there the first
    warning appears.
    Checks also, if relaxation values have been given (or have been forgotten).

\*---------------------------------------------------------------------------*/

#include "checkFieldLimits.H"

// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

bool Foam::checkFieldLimits::check
(
    const volScalarField& field,
    scalar minAllowed,
    scalar maxAllowed,
    bool boundary
) const
{
    bool ok = true;
    scalar min, average, max;
    if(boundary)
    {
        min = Foam::gMin(field);
        average = Foam::gAverage(field);
        max = Foam::gMax(field);
    }
    else
    {
        min = Foam::gMin(field.primitiveField());
        average = Foam::gAverage(field.primitiveField());
        max = Foam::gMax(field.primitiveField());
    }

    Info<< "min mean max ("
	<< field.name()
	<< ":"
	<< field.mesh().name() 
	<< "): "
        << min <<" , "
        << average <<" , "
        << max;

    if(! boundary)
    {
	Info << " (without boundary)";
    }

    if(max > maxAllowed)
    {
	Info << " Warning: too large";
	ok = false;
    }

    if(min < minAllowed)
    {
	Info << " Warning: too low";
	ok = false;
    }

    Info << endl;

    return ok;
}


bool Foam::checkFieldLimits::check
(
    const volVectorField& vField,
    scalar minAllowed,
    scalar maxAllowed,
    bool boundary
) const
{
    volScalarField field(vField.name(), Foam::mag(vField));
    return checkFieldLimits::check(field, minAllowed, maxAllowed, boundary);
}


bool Foam::checkFieldLimits::check
(
    word name,
    const scalarField& field,
    scalar minAllowed,
    scalar maxAllowed
) const
{
    bool ok = true;
    scalar min = Foam::gMin(field);
    scalar average = Foam::gAverage(field);
    scalar max = Foam::gMax(field);

    Info<< "min mean max ("
        << name
        << "): "
        << min <<" , "
        << average <<" , "
        << max;

    if(max > maxAllowed)
    {
        Info << " Warning: too large";
        ok = false;
    }

    if(min < minAllowed)
    {
        Info << " Warning: too low";
        ok = false;
    }

    Info << endl;

    return ok;
}


bool Foam::checkFieldLimits::checkRelax
(
    const volScalarField& field
) const
{
    bool ok = true;

    Info<< "Relaxation of " 
        << field.name() << ":" << field.mesh().name() << " ";

    if(field.mesh().relaxField(field.name()))
    {
	Info<< field.mesh().fieldRelaxationFactor(field.name());
    }
    else
    {
	Info<< "unkown";
	ok = false;
    }
    Info<< " (field), ";

    if(field.mesh().relaxEquation(field.name()))
    {
        Info<< field.mesh().equationRelaxationFactor(field.name());
    }
    else
    {
        Info<< "unkown";
	ok = false;
    }
    Info<< " (eqn)";

    if(!ok)
    {
	Info<< " Warning";
    }
    Info << endl;

    return ok;
}


bool Foam::checkFieldLimits::checkRelax
(
    const volVectorField& field
) const
{
    bool ok = true;

    Info<< "Relaxation of "
        << field.name() << ":" << field.mesh().name() << " ";

    if(field.mesh().relaxField(field.name()))
    {
        Info<< field.mesh().fieldRelaxationFactor(field.name());
    }
    else
    {
        Info<< "unkown";
        ok = false;
    }
    Info<< " (field), ";

    if(field.mesh().relaxEquation(field.name()))
    {
        Info<< field.mesh().equationRelaxationFactor(field.name());
    }
    else
    {
        Info<< "unkown";
        ok = false;
    }
    Info<< " (eqn)";

    if(!ok)
    {
        Info<< " Warning";
    }
    Info << endl;

    return ok;
}


bool Foam::checkFieldLimits::limit
(
    volScalarField& field,
    scalar minAllowed,
    scalar maxAllowed,
    bool boundary
) const
{
    bool limited = false;

    forAll(field, i)
    {
        if(field[i] > maxAllowed)
        {
            field[i] = maxAllowed;	
	    limited = true;
        }
        if(field[i] < minAllowed)
        {
            field[i] = minAllowed;
	    limited = true;
        }
    }

    if(boundary)
    {
	forAll(field.mesh().boundary(), b)
	{
	    fvPatchScalarField& patch = field.boundaryFieldRef()[b];
	    forAll(patch, i)
            {
        	if(patch[i] > maxAllowed)
        	{
            	    patch[i] = maxAllowed;
            	    limited = true;
        	}
        	if(patch[i] < minAllowed)
        	{
            	    patch[i] = minAllowed;
            	    limited = true;
        	}	
	    }
	}
    }

    if(limited)
    {
	Info<< "Warning: limited field " << field.name() << endl;
    }
    return limited;
}

// ************************************************************************* //
