/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::diffusivityModels::fixedDiffusivity

Description
    A fixed value diffusivity model.

    The input dictionary, diffusivity, is of the following form:

    \verbatim
    diffusivity
    {
        type        fixedDiffusivity;
        diff0       diff0 [0 2 -1 0 0 0 0]  2.0e-4;    [m^2/s]
    }
    \endverbatim

\*---------------------------------------------------------------------------*/

#include "fixedDiffusivity.H"
#include "addToRunTimeSelectionTable.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
namespace diffusivityModels
{

defineTypeNameAndDebug(fixedDiffusivity, 0);
addToRunTimeSelectionTable(diffusivityModel, fixedDiffusivity, dictionary);


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

fixedDiffusivity::fixedDiffusivity
(
    const fvMesh& mesh,
    scalarField& diff,
    const labelList& cells,
    const dictionary& dict
)
:
    diffusivityModel(mesh, diff, cells, dict),
    diff0_("diff0", dimensionSet(0,2,-1,0,0,0,0), dict_)
{}


// * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * * //

void fixedDiffusivity::writeData()
{
    Info<< "    diffusivityModels::fixedDiffusivity: " << nl
        << "       diff0 = " << diff0_ << endl;
}


void fixedDiffusivity::evaluate()
{
    forAll(cells_, i)
    {
	diff_[cells_[i]] = diff0_.value();
    }
}

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace diffusivityModels
} // End namespace Foam

// ************************************************************************* //
