/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::materialDatabase

Description
    Provides material properties of a gas.

\*---------------------------------------------------------------------------*/

#ifndef materialDatabase_H
#define materialDatabase_H

#include "fvCFD.H"
#include "fvMesh.H"
#include "dictionary.H"
#include "polyToddYoung.H"
#include "fuelCellSpecies.H"
#include "speciesTable.H"
#include "physicalConstants.H"
#include "diffusivityModels.H"
#include "porousZoneList.H"
#include "checkFieldLimits.H"

namespace Foam
{

/*---------------------------------------------------------------------------*\
                    Class materialDatabase Declaration
\*---------------------------------------------------------------------------*/

class materialDatabase
:
    private checkFieldLimits
{
private:

    // Private data

        //- Reference to mesh
        const fvMesh& mesh_;
        //- Name of the gas (air/fuel)
        const word zName_;
        //- Dict with material properties
        const dictionary speciesDict_;
        //- Table containing the names of the species
        const speciesTable speciesNames_;
        //- List of porous zones
        porousZoneList porousZones_;
    
        //- Mass fraction of each species
        PtrList<volScalarField>& Y_;
        //- Temperature
        const volScalarField& T_;
        //- Velocity
        volVectorField& U_;
        //- Pressure
        const volScalarField& p_; 
        //- Liquid water content
        volScalarField& s_;

        //- Molar fraction of each species
        PtrList<volScalarField> X_;
        //- Molar heat capacity of each species
        PtrList<polyToddYoung> molarCp_;
        //- Dynamic viscosity of each species
        PtrList<polyToddYoung> muS_;
        //- Thermal conductivity of each species
        PtrList<polyToddYoung> kS_;

        //- Diffusivity of the species
        PtrList<volScalarField> diffSp_;

        //- Thermal conductivity of the gas mixture
        volScalarField k_;
        //- Dynamic viscosity of the gas mixture
        volScalarField mu_;
        //- Kinematic viscosity of the gas mixture
        volScalarField nu_;
        //- Density of the gas mixture
        volScalarField rho_;
        //- Heat capacity of the gas mixture
        volScalarField cp_;
        //- Molar heat capacity of the gas mixture
        volScalarField cm_;
        //- Contact angle in the gas area
        volScalarField theta_;
        //- Porosity in the gas area
        volScalarField epsilon0_;
        //- Porous region or not
        volScalarField porous_;
        //- Permeability in the gas area
        volScalarField K0_;

        //- ID of liquid vapour phase
        label H2O_;
        //- Name of inert species (which is not solved for)
        word inertSpecie_;

        //- Used to exchange diffusivities with the diff-models
        scalarField diff_;
        //- List of all species
        PtrList<fuelCellSpecies> species_;
        //- List of all cells of each cell zone
        PtrList<const labelList> znCells_;
        //- List of dictionaries defining the diffusivities
        PtrList<const dictionary> znDiffDict_;
        //- List of diffusivity models
        PtrList <autoPtr<diffusivityModels::diffusivityModel> >
            diffModels_;
        //- Sum of Y_i / M_i
        volScalarField sumYIonMI_;

	//- Exponent for pore blockage due to liquid water
	const scalar expBlockage_;

    // Private member functions

        //- Disallow copy construct
        materialDatabase(const materialDatabase&) = delete;

        //- Disallow default bitwise assignment
        void operator=(const materialDatabase&) = delete;

        //- Limit material properties to some value, e.g. larger 0
        void limit(const scalar min, scalarField& field, const word& name);


public:

    //- Constructor
    materialDatabase
    (	
	PtrList<volScalarField>& Y,
	const volScalarField& T, 
        volVectorField& U,
	const volScalarField& p, 
	volScalarField& s
    );

    //- Destructor
    ~materialDatabase();

    // Member Functions

	//- Update material properties
    	void updateMaterialProperties();

	//- Update mole fractons
    	void updateMoleFractions();

	//- Return reference to thermal conductivity
    	volScalarField& k()
    	{
	    return k_;
    	}

	//- Return reference to dynamic viscosity
  	volScalarField& mu()
    	{
	    return mu_;
    	}

	//- Return reference to kinematic viscosity
    	volScalarField& nu()
    	{
	    return nu_;
    	}

	//- Return reference to density
        volScalarField& rho()
        {
	    return rho_;
        }

	//- Return reference to heat capacity
        volScalarField& cp()
        {
	    return cp_;
        }

	//- Return reference to molar heat capacity
        volScalarField& molarCp()
        {
            return cm_;
        }

	//- Return reference to molar heat capacity model
        PtrList<polyToddYoung>& molarCpModel()
        {
	    return molarCp_;
        }

	//- Return integral of heat capacity between Tr and T
        tmp<scalarField> molarCp(const label i, const scalar Tr, scalarField& T)
        {
	    return molarCp_[i].polyInt(Tr, T);
        }

	//- Return integral of Cp(T)/T dT betweem Tr and T
        tmp<scalarField> molarCpS(const label i, const scalar Tr, scalarField& T)
        {
            return molarCp_[i].polyIntS(Tr, T);
        }

	//- Return integral of heat capacity up to temperature T
        tmp<scalarField> molarCp(const label i, scalarField& T)
        {
            return molarCp_[i].polyInt(T);
        }

	//- Return reference to species diffusivities
        PtrList<volScalarField>& diffusivity()
        {
	    return diffSp_;
        }
	//- Return reference to molar fractions
        PtrList<volScalarField>& X()
        {
	    return X_;
        }

	//- Return ID of vapour
        label waterIndex()
        {
            return H2O_;
        }

    	//- Return contact angle
        volScalarField& theta()
        {
	    return theta_;
        };

	//- Return porosity
        volScalarField& epsilon0()
        {
            return epsilon0_;
        };

	//- Return a field describing whether are is porous or not
        volScalarField& porous()
        {
            return porous_;
        };
	//- Return reference to permeability
        volScalarField& K0()
        {
            return K0_;
        };

	//- Return list of porous zones
        porousZoneList& porousZones()
        {
	    return porousZones_;
        }

	//- Return reference to species list
        PtrList<fuelCellSpecies>& species()
        {
	    return species_;
        }

	//- Return name of inert species
        word& inertSpecie()
        {
	    return inertSpecie_;
        }

	//- Return list of species names
        const speciesTable& speciesNames()
        {
	    return speciesNames_;
        }

	//- Return dictionary with infos about species
        const dictionary& speciesDict()
        {
	    return speciesDict_; 
        }

	//- Return reference to liquid water content
        volScalarField& s()
        {
	    return s_;
        }

	//- Return reference to mass fractions
        PtrList<volScalarField>& Y()
        {
	    return Y_;
        }

	//- Return reference to pressure
        const volScalarField& p()
        {
	    return p_;
        }

	//- Return reference to gas temperature
        const volScalarField& T()
        {
	    return T_;
        }

	//- Return reference to velocity
        volVectorField& U()
        {
	    return U_;
        }

	//- Return reference to mesh
        const fvMesh& mesh()
        {
    	    return mesh_;
        }

	//- Compute thermal conductivity of gas-liquid water mixture
        tmp<scalarField> twoPhaseK(const dimensionedScalar&);

};

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif
