/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::ohmicOverpotentialModel

Description
    A namespace for various ohmic overpotential implementations.

\*---------------------------------------------------------------------------*/

#ifndef ohmicOverpotentialModel_H
#define ohmicOverpotentialModel_H

#include "fvMesh.H"
#include "patchDatabase.H"
#include "dimensionedScalar.H"
#include "IOdictionary.H"
#include "scalarField.H"
#include "surfaceMesh.H"
#include "fvsPatchField.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                Class ohmicOverpotentialModel Declaration
\*---------------------------------------------------------------------------*/

class ohmicOverpotentialModel
{
    // Private data
	//- Reference to electrolyte mesh
	const fvMesh& mesh_;
	//- Database of patch names and IDs
	const patchDatabase& pm_;
	//- Electrolyte thickness
	dimensionedScalar hE_;
	//- Areas specific resistance (fitted or estimated)
	dimensionedScalar R_;
	//- How much can the computed ASR deviate from the fitted/estimated
	//- otherwise the ASR is limited
	scalar deviation_;

    // Private member functions

        //- Disallow copy construct
        ohmicOverpotentialModel(const ohmicOverpotentialModel&);

        //- Disallow default bitwise assignment
        void operator=(const ohmicOverpotentialModel&);

public:

    //- Runtime type information
    TypeName("ohmicOverpotentialModel");

    // Declare run-time constructor selection table

        declareRunTimeSelectionTable
        (
            autoPtr,
            ohmicOverpotentialModel,
            dictionary,
            (
                const fvMesh& mesh,
                const patchDatabase& pm,
                const dictionary& dict
            ),
            (mesh, pm, dict)
        );


    // Selectors

        //- Return a reference to the selected ohmic overpotential model
        static autoPtr<ohmicOverpotentialModel> New
        (
            const fvMesh& mesh,
            const patchDatabase& pm,
            const dictionary& dict
        );


    // Constructors

        ohmicOverpotentialModel
        (
	    const fvMesh& mesh,
	    const patchDatabase& pm,
            const dictionary& dict
        );

    // Destructor

        virtual ~ohmicOverpotentialModel();


    // Member Functions

	//- Return area specific resistance
        virtual tmp<scalarField> ASR
        (
            const scalarField& T,
            const scalarField& lambda
        ) const = 0;

	//- Return electrolyte thickness
	const dimensionedScalar& hE() const
	{
	    return hE_;
	}

	//- Return ASR
        const dimensionedScalar& R() const
	{
	    return R_;
	}

        //- Return allowed deviation of ASR
        scalar deviation() const
	{
	    return deviation_;
	}

	//- Return reference to patch database
	const patchDatabase& pm() const
	{
	    return pm_;
	}

	//- Return reference to electrolyte mesh
	const fvMesh& mesh() const
	{
	    return mesh_;
	}
};

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
