/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::patchDatabase

Description
    Collects all patch names and their IDs, and provides patch to patch
    interpolation for different child meshes.

\*---------------------------------------------------------------------------*/

#include "patchDatabase.H"


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::patchDatabase::patchDatabase
(
    const fvMesh& mesh, 
    const fvMesh& airMesh, 
    const fvMesh& fuelMesh, 
    const fvMesh& electrolyteMesh
)
:
    mesh_(mesh),
    airMesh_(airMesh),
    fuelMesh_(fuelMesh),
    electrolyteMesh_(electrolyteMesh),
    cellProperties_
    (
        IOobject
        (
            "cellProperties",
            mesh_.time().constant(),
            mesh_,
            IOobject::MUST_READ,
            IOobject::NO_WRITE
        )
    ),

    anodeName_(cellProperties_.lookup("anodePatch")),
    fuelInletName_(cellProperties_.lookup("fuelInletPatch")),
    fuelOutletName_(cellProperties_.lookup("fuelOutletPatch")),

    cathodeName_(cellProperties_.lookup("cathodePatch")),
    airInletName_(cellProperties_.lookup("airInletPatch")),
    airOutletName_(cellProperties_.lookup("airOutletPatch")),

    electrolyteAnodeName_(cellProperties_.lookup("electrolyteAnodePatch")),
    electrolyteCathodeName_(cellProperties_.lookup("electrolyteCathodePatch")),

    anodeID_(fuelMesh.boundaryMesh().findPatchID(anodeName_)),
    fuelInletID_(fuelMesh.boundaryMesh().findPatchID(fuelInletName_)),
    fuelOutletID_(fuelMesh.boundaryMesh().findPatchID(fuelOutletName_)),

    cathodeID_(airMesh.boundaryMesh().findPatchID(cathodeName_)),
    airInletID_(airMesh.boundaryMesh().findPatchID(airInletName_)),
    airOutletID_(airMesh.boundaryMesh().findPatchID(airOutletName_)),

    electrolyteAnodeID_(electrolyteMesh.boundaryMesh().findPatchID(electrolyteAnodeName_)),
    electrolyteCathodeID_(electrolyteMesh.boundaryMesh().findPatchID(electrolyteCathodeName_)),
    
    anodePatch_(fuelMesh.boundaryMesh()[anodeID_]),
    cathodePatch_(airMesh.boundaryMesh()[cathodeID_]),
    electrolyteAnodePatch_(electrolyteMesh.boundaryMesh()[electrolyteAnodeID_]),
    electrolyteCathodePatch_(electrolyteMesh.boundaryMesh()[electrolyteCathodeID_]),

    anodeToCathode_(anodePatch_,cathodePatch_),
    cathodeToAnode_(cathodePatch_,anodePatch_),
    fuelAnodeToElectrolyteAnode_(fuelMesh.boundaryMesh()[anodeID_],electrolyteMesh.boundaryMesh()[electrolyteAnodeID_]),
    airCathodeToElectrolyteCathode_(airMesh.boundaryMesh()[cathodeID_],electrolyteMesh.boundaryMesh()[electrolyteCathodeID_]),
    electrolyteAnodeToFuelAnode_(electrolyteMesh.boundaryMesh()[electrolyteAnodeID_],fuelMesh.boundaryMesh()[anodeID_])
{
    Info<< nl << "Setting special patch IDs\n" << endl;
    Info<< "    anodeName              = " << anodeName_ << nl
        << "    fuelInletName          = " << fuelInletName_ << nl
        << "    fuelOutletName         = " << fuelOutletName_ << nl
        << "    cathodeName            = " << cathodeName_ << nl
        << "    airInletName           = " << airInletName_ << nl
        << "    airOutletName          = " << airOutletName_ << nl
        << "    electrolyteAnodeName   = " << electrolyteAnodeName_ << nl
        << "    electrolyteCathodeName = " << electrolyteCathodeName_ << nl
        << endl;

    // check, if patches have been found
    if (anodeID_ == -1)
    {
        FatalErrorInFunction
            << "Cannot find patch " << anodeName_ << " on the fuel mesh."
            << abort(FatalError);
    }
    if (fuelInletID_ == -1)
    {
        FatalErrorInFunction
            << "Cannot find patch " << fuelInletName_ << " on the fuel mesh."
            << abort(FatalError);
    }
    if (fuelOutletID_ == -1)
    {
        FatalErrorInFunction
            << "Cannot find patch " << fuelOutletName_ << " on the fuel mesh."
            << abort(FatalError);
    }
    if (cathodeID_ == -1)
    {
        FatalErrorInFunction
            << "Cannot find patch " << cathodeName_ << " on the air mesh."
            << abort(FatalError);
    }
    if (airInletID_ == -1)
    {
        FatalErrorInFunction
            << "Cannot find patch " << airInletName_ << " on the air mesh."
            << abort(FatalError);
    }
    if (airOutletID_ == -1)
    {
        FatalErrorInFunction
            << "Cannot find patch " << airOutletName_ << " on the air mesh."
            << abort(FatalError);
    }
    if (electrolyteAnodeID_ == -1)
    {
        FatalErrorInFunction
            << "Cannot find patch " << electrolyteAnodeName_
            << " on the electrolyte mesh."
            << abort(FatalError);
    }
    if (electrolyteCathodeID_ == -1)
    {
        FatalErrorInFunction
            << "Cannot find patch " << electrolyteAnodeName_
            << " on the electrolyte mesh."
            << abort(FatalError);
    }

    // check, if some patches have the same size
    if (electrolyteAnodePatch_.size() != electrolyteCathodePatch_.size())
    {
        FatalError
            << "Anode and cathode patch do not have the same size." <<nl
            << exit (FatalError);
    }

    // sometimes OpenFOAM does not find the faces for projection
    // this will lead to a warning here, because then the distance between the two faces will be very large
    // this happens especially, if the projecting faces are on different processors (avoid that!)
    check("anodeToCathode interpolation", anodeToCathode_.faceDistanceToIntersection(), SMALL, 1e-3);
    check("cathodeToAnode interpolation", cathodeToAnode_.faceDistanceToIntersection(), SMALL, 1e-3);
    check("fuelAnodeToElectrolyteAnode interpolation", fuelAnodeToElectrolyteAnode_.faceDistanceToIntersection(), -SMALL, SMALL);
    check("airCathodeToElectrolyteCathode interpolation", airCathodeToElectrolyteCathode_.faceDistanceToIntersection(), -SMALL, SMALL);
    check("electrolyteAnodeToFuelAnode interpolation", electrolyteAnodeToFuelAnode_.faceDistanceToIntersection(), -SMALL, SMALL);
}

// ************************************************************************* //
