/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::phaseChange

Description
    Computes phase change rate depending on the partial pressure.

\*---------------------------------------------------------------------------*/

#include "phaseChange.H"


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::phaseChange::phaseChange
(
    const IOdictionary& dict,
    materialDatabase& mat
)
:
    T0_(dimensionedScalar("T0", dimTemperature, 273.15)),
    gammaCond_("gammaCondensation", dimless/dimTime, dict.subDict("phaseChange")),
    gammaEvap_("gammaEvaporation", dimless/dimTime, dict.subDict("phaseChange")),
    sMax_(dict.subDict("phaseChange").lookupOrDefault<scalar>("sMax",0.99)),
    phaseChangeOn_(dict.subDict("phaseChange").lookupOrDefault<bool>("active",true)),
    T_(mat.T()),
    mat_(mat),
    Svl_
    (
        IOobject
        (
            "Svl",
            mat.mesh().time().timeName(),
            mat.mesh(),
            IOobject::READ_IF_PRESENT,
            IOobject::AUTO_WRITE
        ),
        mat.mesh(),
        dimensionedScalar("0", dimensionSet(1,-3,-1,0,0,0,0), 0),
        zeroGradientFvPatchScalarField::typeName
    ),
    pSat_
    (
        IOobject
        (
            "pSat",
            mat.T().mesh().time().timeName(),
            mat.T().mesh(),
            IOobject::READ_IF_PRESENT,
            IOobject::AUTO_WRITE
        ),
        mat.T().mesh(),
        dimensionedScalar("0", dimPressure, 0.0)
    ),
    partialP_        
    (
        IOobject
        (
            "partialPressure",
            mat.T().mesh().time().timeName(),
            mat.T().mesh(),
            IOobject::NO_READ,
            IOobject::NO_WRITE
        ),
        mat.T().mesh(),
        dimensionedScalar("0", dimPressure, 0.0)
    ),
    humidity_        
    (
        IOobject
        (
            "humidity",
            mat.T().mesh().time().timeName(),
            mat.T().mesh(),
            IOobject::READ_IF_PRESENT,
            IOobject::AUTO_WRITE
        ),
        mat.T().mesh(),
        dimensionedScalar("0", dimless, 0.0)
    )
{
    if(phaseChangeOn_ == false)
    {
        Info<< "Warning: phase change switched off." << endl;
    }
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void phaseChange::solve()
{
    // get temperature in degr celsius
    volScalarField TC = T_ - T0_;

    // get saturation pressures
    pSat_ = 
	dimensionedScalar("p", dimensionSet(1,-1,-2,0,0,0,0), 101325.)
	*pow(10, -2.1794 + dimensionedScalar("0", dimensionSet(0,0,0,-1,0,0,0), 0.02953)*TC
	- dimensionedScalar("0", dimensionSet(0,0,0,-2,0,0,0), 9.1837e-5)*TC*TC 
	+ dimensionedScalar("0", dimensionSet(0,0,0,-3,0,0,0), 1.4454e-7)*TC*TC*TC);

    // partial pressure H2O
    partialP_ = mat_.p()*mat_.X()[mat_.waterIndex()];

    // get phase change source term (vapour to liquid)
    forAll(Svl_,i)
    {
        if(partialP_[i] >= pSat_[i])
        {
            Svl_[i] = 
		(gammaCond_.value()*(sMax_ - mat_.s()[i])*mat_.epsilon0()[i]
		*mat_.species()[mat_.waterIndex()].MW()
		*(mat_.p()[i]*mat_.X()[mat_.waterIndex()][i] - pSat_[i])
		/(physicalConstant::Rgas*T_[i]*1000.)).value(); 
		// 1000. because MW comes in kg/kmol
        }
        else
        {
            Svl_[i] = 
		(gammaEvap_.value()*mat_.s()[i]*mat_.epsilon0()[i]
		*mat_.species()[mat_.waterIndex()].MW()
		*(mat_.p()[i]*mat_.X()[mat_.waterIndex()][i] - pSat_[i])
		/(physicalConstant::Rgas*T_[i]*1000.)).value();
        }
    }

    humidity_ = 100.0*partialP_/pSat_;

    check(pSat_, 0, GREAT, false);
    check(partialP_, 0, GREAT, false);
    check(humidity_, 0, 100.0, false);
    check("liquid water source:" + T_.mesh().name(), Svl_, -GREAT, GREAT);

    // set phase change to zero, if necessary
    if(phaseChangeOn_ == false)
    {
        Svl_  = 0.0;
    }
}

// ************************************************************************* //
