/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::polyToddYoung

Description
    Polynomial function for different material properties. Coefficients
    will be read from file

\*---------------------------------------------------------------------------*/

#ifndef polyToddYoung_H
#define polyToddYoung_H

#include "scalar.H"
#include "scalarField.H"
#include "FixedList.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                           Class polyToddYoung Declaration
\*---------------------------------------------------------------------------*/

class polyToddYoung
{
    // Private data
  	//- Coefficients for polynomial function
        scalar a0_, a1_, a2_, a3_, a4_, a5_, a6_;


    // Private member functions

        //- Disallow default bitwise assignment
        void operator=(const polyToddYoung&) = delete;

public:

    // Public data
	//- Number of coefficients
        static const int nCoeffs_ = 7;
        typedef FixedList<scalar, nCoeffs_> coeffArray;


    // Constructors

        //- Construct from coefficients
        polyToddYoung
        (
            scalar a0, scalar a1, scalar a2,
            scalar a3, scalar a4, scalar a5, scalar a6
        );

        //- Construct from Istream
        polyToddYoung(Istream& is);

        //- Construct from dictionary
        polyToddYoung(const dictionary& dict);

        //- Construct as a named copy
        inline polyToddYoung(const word&, const polyToddYoung&);


    // Destructor

        ~polyToddYoung()
        {}


    // Member Functions

        //- Evaluate the polynomial function and return the result
        scalar polyVal(scalar T);

        //- Evaluate the polynomial function and return the result
        tmp<scalarField> polyVal(scalarField T);


        // in case of Cp(T) polynomial, integral of Cp(T)dT useful for enthalpy
        //- Evaluate the integral of the polynomial function and return the result
        scalar polyInt(scalar T);

        //- Evaluate the integral of the polynomial function and return the result
        tmp<scalarField> polyInt(scalarField T);

        //- Evaluate definite integral from Tlow to Thigh and return result
        scalar polyInt(scalar Tlow, scalar Thigh);

        //- Evaluate definite integral from Tlow to Thigh and return result
        tmp<scalarField> polyInt(scalar Tlow, scalarField Thigh);

        //- Evaluate definite integral from Tlow to Thigh and return result
        tmp<scalarField> polyInt(scalarField Tlow, scalarField Thigh);


        // in case of Cp(T) polynomial, integral of Cp(T)/T dT useful for entropy
        //- Evaluate the integral of the polynomial function divided by T
        //  and return the result
        scalar polyIntS(scalar T);
        tmp<scalarField> polyIntS(scalarField T);

        //- Evaluate definite integral from Tlow to Thigh
        //  of the polynomial divided by T and return result
        scalar polyIntS(scalar Tlow, scalar Thigh);
        tmp<scalarField> polyIntS(scalar Tlow, scalarField Thigh);
        tmp<scalarField> polyIntS(scalarField Tlow, scalarField Thigh);


        //- Write the function coefficients
        void writeData();
        void writeData(Ostream& os);
};

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
