/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::solidZone

Description
    Top level model for solid zones. Provides the properties of these
    solid zones.

\*---------------------------------------------------------------------------*/

#ifndef solidZone_H
#define solidZone_H

#include "fvMesh.H"
#include "dictionary.H"
#include "fvMatricesFwd.H"
#include "runTimeSelectionTables.H"
#include "coordinateSystem.H"
#include "dimensionedVector.H"
#include "keyType.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                        Class solidZone Declaration
\*---------------------------------------------------------------------------*/

class solidZone
:
    public regIOobject
{
    // Private Member Functions

        //- Disallow default bitwise copy construct
        solidZone(const solidZone&);

        //- Disallow default bitwise assignment
        void operator=(const solidZone&);


protected:

    // Protected data

        //- Reference to the mesh database
        const fvMesh& mesh_;

        //- Dictionary used for model construction
        const dictionary dict_;

        //- Name(s) of cell-zone
        keyType zoneName_;

        //- Cell zone IDs
        labelList cellZoneIDs_;

	//- Thermal conductivity
        dimensionedScalar k_;
	//- Heat capacity
	dimensionedScalar cp_;
	//- Density
	dimensionedScalar rho_;


public:

    //- Runtime type information
    TypeName("solidZone");


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            autoPtr,
            solidZone,
            mesh,
            (
                const fvMesh& mesh,
                const dictionary& dict,
                const word& cellZoneName
            ),
            (mesh, dict, cellZoneName)
        );


    // Constructors

        solidZone
        (
            const fvMesh& mesh,
            const dictionary& dict,
            const word& cellZoneName = word::null
        );

        //- Return pointer to new solidZone object created on the freestore
        //  from an Istream
        class iNew
        {
            //- Reference to the mesh database
            const fvMesh& mesh_;

        public:

            iNew
            (
                const fvMesh& mesh
            )
            :
                mesh_(mesh)
            {}

            autoPtr<solidZone> operator()(Istream& is) const
            {
                const dictionary dict(is);

                return autoPtr<solidZone>
                (
                    solidZone::New(mesh_, dict)
                );
            }
        };


    // Selectors

        static autoPtr<solidZone> New
        (
            const fvMesh& mesh,
            const dictionary& dict,
            const word& cellZoneName = word::null
        );


    //- Destructor
    virtual ~solidZone();


    // Member Functions

        //- Return const access to the zone name
        inline const keyType& zoneName() const;

        //- Return const access to the cell zone IDs
        inline const labelList& zoneIds() const;

	//- Return thermal conductivity
        dimensionedScalar k() const;
	//- Return heat capacity
        dimensionedScalar cp() const;
	//- Return density
        dimensionedScalar rho() const;

        //- Write
        virtual bool writeData(Ostream& os) const;

        //- Inherit read from regIOobject
        using regIOobject::read;

        //- Read solid dictionary
        virtual bool read(const dictionary& dict);
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#include "solidZoneI.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
