/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2022 Fahlbeck, Nilsson, Salehi., Chalmers University of Technology
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "headLossPressureFvPatchScalarField.H"
#include "addToRunTimeSelectionTable.H"
#include "fvPatchFieldMapper.H"
#include "volFields.H"
#include "surfaceFields.H"
#include "gravityMeshObject.H"
// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //
Foam::headLossPressureFvPatchScalarField::headLossPressureFvPatchScalarField
(
    const fvPatch& p,
    const DimensionedField<scalar, volMesh>& iF
)
:
    fixedValueFvPatchScalarField(p, iF),
    UName_("U"),
    phiName_("phi"),
    pFar_(p.size(), Zero),
    dP_(),
    g_(),
    frictionLossFactors_(),
    minorLossFactors_(),
    HFar_(),
    kDynamic_(),
    dkDynamic_(),
    flowRate_(),
    Ar_(),
    tOld_(),
    sumPhi_(),
    dFar_(),
    UFarScale_(),
    Tol_(),
    Nitr_(),
    fDiff_()
{}

Foam::headLossPressureFvPatchScalarField::headLossPressureFvPatchScalarField
(
    const fvPatch& p,
    const DimensionedField<scalar, volMesh>& iF,
    const dictionary& dict
)
:
    fixedValueFvPatchScalarField(p, iF, dict, false),
    UName_(dict.lookupOrDefault<word>("U", "U")),
    phiName_(dict.lookupOrDefault<word>("phi", "phi")),
    pFar_("pFar", dict, p.size()),
    dP_(dict.get<scalar>("dP")),
    g_(dict.getOrDefault<scalar>("g",9.81)),
    frictionLossFactors_(dict.lookup("frictionLossFactors")),
    minorLossFactors_(dict.lookup("minorLossFactors")),
    HFar_(dict.get<scalar>("HFar")),
    dFar_(dict.getOrDefault<scalar>("dFar",0)),
    Tol_(dict.getOrDefault<scalar>("Tol",1e-6)),
    Nitr_(dict.getOrDefault<scalar>("Nitr",20)),
    fDiff_(dict.getOrDefault<scalar>("fDiff",0.05))
{
    if (dict.found("value"))
    {
        fvPatchField<scalar>::operator=
        (
            scalarField("value", dict, p.size())
        );
    }
    else
    {
        fvPatchField<scalar>::operator=(pFar_);
    }

    if (dict.found("kDynamic"))
    {
        kDynamic_=(Function1<scalar>::New("kDynamic", dict));
        dkDynamic_=(dict.get<scalar>("dkDynamic"));
    }

    if (dict.found("flowRate"))
    {
        flowRate_=(Function1<scalar>::New("flowRate",dict));
        Ar_=(dict.get<scalar>("Ar"));
    }

    // UFarScale to calculate the UFal velocity with dFar
    UFarScale_ = (dFar_ != 0) ? pow(dP_/dFar_,4): 0;

    tOld_=db().time().value();

    // Check to verify that diameter of the patch is not 0
    if (dP_==0)
    {
        FatalErrorInFunction
            << " The diameter dP of patch " << this->patch().name()
            << " is specified as 0" << nl
            << " This is not allowed! " << nl
            << " Please specify the correct diameter of the patch"
            << " of field " << this->internalField().name() << nl
            << " in file " << this->internalField().objectPath()
            << exit(FatalError);
    }
    
    // Check to verify that diameter is specified for each loss
    forAll(frictionLossFactors_, i)
    {
        if (frictionLossFactors_[i].first().component(0)==0)
        {
            FatalErrorInFunction
                << " The diameter of frictionLossFactors " << frictionLossFactors_[i].second()
                << " is specified as 0" << nl
                << " This is not allowed! " << nl
                << " Please specify a correct diameter for the head loss" << nl
                << " or if not using frictionLossFactors, specify an arbitrary diameter" << nl
                << " on patch " << this->patch().name()
                << " of field " << this->internalField().name()
                << " in file " << this->internalField().objectPath()
                << exit(FatalError);
        }
    }

    // Check to verify that diameter is specified for each loss
    forAll(minorLossFactors_, i)
    {
        if (minorLossFactors_[i].first().component(0)==0)
        {
            FatalErrorInFunction
                << " The diameter of minorLossFactors " << minorLossFactors_[i].second()
                << " is specified as 0" << nl
                << " This is not allowed! " << nl
                << " Please specify a correct diameter for the head loss" << nl
                << " or if not using minorLossFactors, specify an arbitrary diameter" << nl
                << " on patch " << this->patch().name()
                << " of field " << this->internalField().name()
                << " in file " << this->internalField().objectPath()
                << exit(FatalError);
        } 
    }
}

Foam::headLossPressureFvPatchScalarField::headLossPressureFvPatchScalarField
(
    const headLossPressureFvPatchScalarField& ptf,
    const fvPatch& p,
    const DimensionedField<scalar, volMesh>& iF,
    const fvPatchFieldMapper& mapper
)
:
    fixedValueFvPatchScalarField(ptf, p, iF, mapper),
    UName_(ptf.UName_),
    phiName_(ptf.phiName_),
    pFar_(ptf.pFar_, mapper),
    dP_(ptf.dP_),
    g_(ptf.g_),
    frictionLossFactors_(ptf.frictionLossFactors_),
    minorLossFactors_(ptf.minorLossFactors_),
    HFar_(ptf.HFar_),
    kDynamic_(ptf.kDynamic_.clone()),
    dkDynamic_(ptf.dkDynamic_),
    flowRate_(ptf.flowRate_.clone()),
    Ar_(ptf.Ar_),
    tOld_(ptf.tOld_),
    sumPhi_(ptf.sumPhi_),
    dFar_(ptf.dFar_),
    UFarScale_(ptf.UFarScale_),    
    Tol_(ptf.Tol_),
    Nitr_(ptf.Nitr_),
    fDiff_(ptf.fDiff_)
{}


Foam::headLossPressureFvPatchScalarField::headLossPressureFvPatchScalarField
(
    const headLossPressureFvPatchScalarField& tppsf
)
:
    fixedValueFvPatchScalarField(tppsf),
    UName_(tppsf.UName_),
    phiName_(tppsf.phiName_),
    pFar_(tppsf.pFar_),
    dP_(tppsf.dP_),
    g_(tppsf.g_),
    frictionLossFactors_(tppsf.frictionLossFactors_),
    minorLossFactors_(tppsf.minorLossFactors_),
    HFar_(tppsf.HFar_),
    kDynamic_(tppsf.kDynamic_.clone()),
    dkDynamic_(tppsf.dkDynamic_),
    flowRate_(tppsf.flowRate_.clone()),
    Ar_(tppsf.Ar_),
    tOld_(tppsf.tOld_),
    sumPhi_(tppsf.sumPhi_),
    dFar_(tppsf.dFar_),
    UFarScale_(tppsf.UFarScale_),
    Tol_(tppsf.Tol_),
    Nitr_(tppsf.Nitr_),
    fDiff_(tppsf.fDiff_)
{}


Foam::headLossPressureFvPatchScalarField::headLossPressureFvPatchScalarField
(
    const headLossPressureFvPatchScalarField& tppsf,
    const DimensionedField<scalar, volMesh>& iF
)
:
    fixedValueFvPatchScalarField(tppsf, iF),
    UName_(tppsf.UName_),
    phiName_(tppsf.phiName_),
    pFar_(tppsf.pFar_),
    dP_(tppsf.dP_),
    g_(tppsf.g_),
    frictionLossFactors_(tppsf.frictionLossFactors_),
    minorLossFactors_(tppsf.minorLossFactors_),
    HFar_(tppsf.HFar_),
    kDynamic_(tppsf.kDynamic_.clone()),
    dkDynamic_(tppsf.dkDynamic_),
    flowRate_(tppsf.flowRate_.clone()),
    Ar_(tppsf.Ar_),
    tOld_(tppsf.tOld_),
    sumPhi_(tppsf.sumPhi_),
    dFar_(tppsf.dFar_),
    UFarScale_(tppsf.UFarScale_),
    Tol_(tppsf.Tol_),
    Nitr_(tppsf.Nitr_),
    fDiff_(tppsf.fDiff_)
{}

// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::headLossPressureFvPatchScalarField::autoMap
(
    const fvPatchFieldMapper& m
)
{
    fixedValueFvPatchScalarField::autoMap(m);
    pFar_.autoMap(m);
}


void Foam::headLossPressureFvPatchScalarField::rmap
(
    const fvPatchScalarField& ptf,
    const labelList& addr
)
{
    fixedValueFvPatchScalarField::rmap(ptf, addr);

    const headLossPressureFvPatchScalarField& tiptf =
        refCast<const headLossPressureFvPatchScalarField>(ptf);

    pFar_.rmap(tiptf.pFar_, addr);
}

Foam::scalar Foam::headLossPressureFvPatchScalarField::fFunc
(
    const scalar& diameterLoss,
    const scalar& surfaceRoughness,
    const scalar& UPatch,
    const scalar& nu,
    const label& frictionLossFactorsIndex
)
{
    const scalar& ULoss = UPatch*sqr(dP_/diameterLoss);
    const scalar& Re = diameterLoss*ULoss/nu;

    // Generate additional information about friction loss coefficient
    if (debug)
    {
        Info << "Friction coefficient for frictionLossFactors "
             << frictionLossFactors_[frictionLossFactorsIndex].second()
             << " on patch " << patch().name() << ":" << nl
             << "    The Re number is " << Re << " and the flow is ";
    }

    scalar f(0);
    if ( Re == 0 || sumPhi_ == VSMALL )
    {
        f = 0;
        if (debug)
        {
            Info << "zero" << nl
            << "    Friction coefficient has the value of " << f << nl << endl;
        }
    }
    else if ( Re < 2300 )
    {
        f = 64/Re;
        if (debug)
        {
            Info << "laminar" << nl
            << "    Friction coefficient has the value of " << f << nl << endl;
        }
    }
    else
    {
        const scalar fInitial = 
            sqr(1/(-1.8*log10(6.9/Re + pow((surfaceRoughness/diameterLoss)/3.7 ,1.11))));
        scalar fOld = fInitial;
        for(int i = 0; i < Nitr_; i++)
        {
            f = sqr(1/(-2*log10((surfaceRoughness/diameterLoss)/3.7 + 2.51/(Re*sqrt(fOld)))));
            if (mag(f - fOld)/fOld <= Tol_ )
            {
                if (debug)
                {
                    Info << "turbulent" << nl
                         << "    Friction coefficient converge to the value of " << f << nl
                         << "    The number of iterations " << i << nl
                         << "    The iteration procedure reached a tolerance of " << mag(f - fOld)/fOld 
                         << nl << endl;
                }
                break;
            }
            fOld = f;
        }
        if ( mag(f - fInitial)/fInitial >= fDiff_ )
        {
            Info << "Using f initial for frictionLossFactors " 
                 << frictionLossFactors_[frictionLossFactorsIndex].second() 
                 << " on patch " << patch().name() << endl;
            f = fInitial;
        }
    }
    return f;
}


void Foam::headLossPressureFvPatchScalarField::updateCoeffs
(
    const scalarField& pFar,
    const vectorField& Up
)
{
    if (updated())
    {
        return;
    }

    const fvsPatchField<scalar>& phip =
        patch().lookupPatchField<surfaceScalarField, scalar>(phiName_);

    // Update HFar
    // This is placed before update sumPhi since we want phi from last time step.
    if (flowRate_ && tOld_<db().time().value())
    {
        scalar dH = (flowRate_->value(this->db().time().timeOutputValue()) + sumPhi_)/Ar_*db().time().deltaTValue();
        HFar_ = HFar_ + dH;
        Info << "H = " << HFar_ << " m" << endl;
    }

    // Calculating the Average velocity on the patch, Uavg = phi/A
    sumPhi_ = gSum(patch().lookupPatchField<surfaceScalarField, scalar>(phiName_));
    const scalar& totalArea = gSum(patch().magSf());
    scalar Uavg;
    if (sumPhi_ != 0)
    {
        Uavg = mag(sumPhi_)/totalArea;
    }
    else
    {
        Info << "Zero flux on patch" << endl;
        sumPhi_ = VSMALL;
        Uavg = mag(sumPhi_)/totalArea;
    }

    // Give warning if reverse flow is detected on patch
    if ( (sign(sumPhi_)!=sign(gMin(phip)+VSMALL) || sign(sumPhi_)!=sign(gMax(phip)-VSMALL)) && sumPhi_!=VSMALL)
    {
        Info << "Warning, reverse flow detected on patch " << patch().name() << endl;
    }

    // Present additional information about the reverse flow
    if (debug)
    {
        if ( (sign(sumPhi_)!=sign(gMin(phip)+VSMALL) || sign(sumPhi_)!=sign(gMax(phip)-VSMALL)) && sumPhi_!=VSMALL)
        {
            scalar nCells = returnReduce(patch().size(), sumOp<label>());
            scalar nReverse = (sign(sumPhi_) == 1) ? mag(gSum(neg(phip))) : mag(gSum(pos(phip))); 
            scalar sumMagPhi = gSum(mag(phip));
            scalar sumReversePhi = min(mag(gSum(neg(phip)*phip)), mag(gSum(pos(phip)*phip)));
            Info << "    Reverse flow on " << nReverse << " out of " << nCells << " cells" << nl
                 << "    Reverse flow is " << sumReversePhi/sumMagPhi*100 
                 << " % of the total flow rate" << nl << endl;
        }
    }

    // Calculating the Minor losses
    scalar kDynamic = kDynamic_ ? kDynamic_->value(this->db().time().timeOutputValue()) : 0;
    scalar dpMinor = kDynamic ? kDynamic*sqr(Uavg*sqr(dP_/dkDynamic_))/2 : 0;
    scalar dMinor;
    scalar k;
    forAll(minorLossFactors_, i)
    {
        dMinor = minorLossFactors_[i].first().component(0);
        k = minorLossFactors_[i].first().component(1);
        dpMinor += k * sqr(Uavg * sqr(dP_ / dMinor))/2;
    }

    // Kinematic viscosity from patch
    const scalar& nu = gAverage(patch().lookupPatchField<volScalarField, scalar>("nu"));

    // Calculating the Friction Losses
    scalar f(0);
    scalar dpFriction(0);
    scalar dFriction;
    scalar epsilon;
    scalar L;
    forAll(frictionLossFactors_, i)
    {
        dFriction = frictionLossFactors_[i].first().component(0);
        epsilon = frictionLossFactors_[i].first().component(1);
        L = frictionLossFactors_[i].first().component(2);
        f = fFunc(dFriction,epsilon,Uavg,nu,i);
        dpFriction += f * L / dFriction * sqr(Uavg * sqr(dP_  / dFriction))/2;
    }

    scalar gHFar;

    // Calculate gHFar via checkinf if gravity based solver or not.
    const wordList& avalibleUDVF = db().time().names<uniformDimensionedVectorField>();     
    if (std::find(avalibleUDVF.begin(), avalibleUDVF.end(), "g") != avalibleUDVF.end())
    {
        const uniformDimensionedVectorField& g =
            meshObjects::gravity::New(db().time());

        gHFar = HFar_ * mag(g.value() & (cmptMag(g.value())/mag(g.value()))); // HFar expressed in relation to hRef

        // Print message at first times step, in the first corrector loop 
        if ( db().time().startTime().value()+db().time().deltaTValue() >= db().time().value() && tOld_<db().time().value() )
        {
            const uniformDimensionedScalarField& hRef =
                db().lookupObject<uniformDimensionedScalarField>("hRef");

            dimensionedScalar ghRef =
            (
                mag(g.value()) > SMALL
                ? g & (cmptMag(g.value())/mag(g.value()))*hRef
                : dimensionedScalar("ghRef", g.dimensions()*dimLength, 0)
            );
            
            const scalarField ghPatch(ghRef.value()-(g.value() & patch().Cf())); // position of patch
            Info << patch().name() << ":" << endl;
            Info << "HPatch = " << gAverage(ghPatch)/mag(g.value()) << " m" << endl;
            Info << "HFar = " << gHFar/mag(g.value()) << " m" << endl;
            Info << "dH = " << gAverage(gHFar-ghPatch)/mag(g.value()) << " m" << endl;
            Info << endl;
        }
    }
    else
    {
        gHFar = HFar_*g_;
    }

    // Update tOld_
    tOld_ = db().time().value(); 

    if (internalField().dimensions() == dimPressure)
    {
        // Pressure as pressure, this BC should only be used for incompressible flow
        operator==
        (
            pFar + (gHFar + sumPhi_/mag(sumPhi_)*(dpFriction + dpMinor)
            - 0.5*neg(phip)*magSqr(Up) - 0.5*pos(sumPhi_)*magSqr(Uavg)
            + 0.5*UFarScale_*magSqr(Uavg))
            *patch().lookupPatchField<volScalarField, scalar>("rho")
        );
    }
    else if (internalField().dimensions() == dimPressure/dimDensity)
    {
        // Pressure as kinematic pressure
        operator==
        (
            pFar + gHFar + sumPhi_/mag(sumPhi_)*(dpFriction + dpMinor)
            - 0.5*neg(phip)*magSqr(Up) - 0.5*pos(sumPhi_)*magSqr(Uavg)
            + 0.5*UFarScale_*magSqr(Uavg) 
        );

    }
    else
    {
        FatalErrorInFunction
            << " Incorrect pressure dimensions " << internalField().dimensions()
            << nl
            << "    Should be " << dimPressure
            << " for compressible/variable density flow" << nl
            << "    or " << dimPressure/dimDensity
            << " for incompressible flow," << nl
            << "    on patch " << this->patch().name()
            << " of field " << this->internalField().name()
            << " in file " << this->internalField().objectPath()
            << exit(FatalError);
    }

    fixedValueFvPatchScalarField::updateCoeffs();
}


void Foam::headLossPressureFvPatchScalarField::updateCoeffs()
{
    updateCoeffs
    (
        pFar(),
        patch().lookupPatchField<volVectorField, vector>(UName())
    );
}


void Foam::headLossPressureFvPatchScalarField::write(Ostream& os) const
{
    fvPatchScalarField::write(os);
    os.writeEntryIfDifferent<word>("U", "U", UName_);
    os.writeEntryIfDifferent<word>("phi", "phi", phiName_);
    pFar_.writeEntry("pFar", os);
    os.writeEntry<scalar>("HFar", HFar_);
    os.writeEntry<scalar>("dP", dP_);
    os.writeEntryIfDifferent<scalar>("g", 9.81,g_);
    os.writeEntry("frictionLossFactors",frictionLossFactors_);
    os.writeEntry("minorLossFactors",minorLossFactors_);
    if (kDynamic_)
    {
        kDynamic_->writeData(os);
        os.writeEntry<scalar>("dkDynamic", dkDynamic_);
    }
    if (flowRate_)
    {
        flowRate_->writeData(os);
        os.writeEntry<scalar>("Ar", Ar_);
    }
    os.writeEntryIfDifferent<scalar>("Tol", 1e-6,Tol_); 
    os.writeEntryIfDifferent<scalar>("Nitr", 20,Nitr_); 
    os.writeEntryIfDifferent<scalar>("fDiff", 0.05,fDiff_);   
    writeEntry("value", os);
}

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
    makePatchTypeField
    (
        fvPatchScalarField,
        headLossPressureFvPatchScalarField
    );
}

// ************************************************************************* //
