/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2022 Fahlbeck, Nilsson, Salehi., Chalmers University of Technology
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::headLossPressureFvPatchScalarField

Group
    grpInletBoundaryConditions grpOutletBoundaryConditions

Description
    This boundary condition provides an incompressible (kinematic) pressure condition for inflow
    and outlow patches with head losses and surface elevation included.

     Reference:
     \verbatim
         Fahlbeck, J., Nilsson, H., & Salehi, S. (2022).
         A Head Loss Pressure Boundary Condition for Hydraulic Systems.
         OpenFOAM Journal, Volume 2, 1-12.
         DOI:10.51560/ofj.v2.69
         ISSN:2753-8168
     \endverbatim

    1. For incompressible pressure for inflow:
        \f[
            p_p = pFar + g*HFar - dp_loss - (Up^2 - uFar^2)/2
        \f]
    2. For incompressible pressure for outflow:
        \f[
            p_p = pFar + g*HFar + dp_loss - (uAvg^2 - uFar^2)/2
        \f]
        where
        \vartable
            p_p     | Kinematic pressure at patch [m2/s2]
            pFar    | Kinematic pressure far from patch [m2/s2]
            Up      | Velocity on face i on patch [m/s]
            uAvg    | Average velocity on patch (phi/area) [m/s]
            u_Far   | Velocity far from patch (calculated from U_avg) [m/s]
            dp_loss | Kinematic pressure losses due to minor and friction effects [m2/s2]
            HFar    | Elevation far from the patch [m]
            g       | Gravitational acceleration [kg m/s2]
        \endvartable

    The loss factor dp_loss is based on steady state loss factors and calculated as:
        \f[
            dp_loss = sum(f_i*L_i/d_i * 0.5 |u_i|^2) + sum(k_j *0.5 |u_j|^2)
        \f]
    Here f is the friction factor, L is pipe length, d is diameter of the pipe,
    k is the minor loss coefficient. Indices i and j are for the ith and jth friction
    and minor loss, respectively . f is directly solved in a laminar case as:
        \f[
            f = 64/Re
        \f]
    and iteratively via the Colebrook equation in a turbulent case as:
        \f[
            1/sqrt(f) = -2.0 log10( (epsilon/d)/3.7 + 2.51/(Re*sqrt(f)))
        \f[
    Here epsilon is the surface roughness and Re the Reynolds number. The switch 
    between turbulent and laminar is assumed to be at Re = 2300, below laminar
    and above turbulent.


Usage
    \table
        Property           | Description                      | Required | Dimensions | Default | Function1
        pFar               | Pressure far                     | yes      | m^2/s^2    | -       | no
        HFar               | Elevation far from patch         | yes      | m          | -       | no
        dP                 | Hydraulic diameter of patch      | yes      | m          | -       | no
        minorLossFactors   | Minor loss factors               | yes      | (m,-), -   | -       | no
        frictionLossFactors| Friction loss factors            | yes      | (m,m,m), - | -       | no
        kDynamic           | Dynamic minor loss coefficient   | no       | -          | -       | yes
        dkDynamic          | Hydraulic diameter of kDynamic   | no       | m          | -       | no
        flowRate           | Flowrate to the reservoir        | no       | m^3/s      | -       | yes
        Ar                 | Reservoir surface area           | no       | m^2        | -       | no
        dFar               | Hydraulic diameter far from patch| no       | m          | 0       | no
        U                  | Velocity field name              | no       | -          | U       | no
        phi                | Flux field name                  | no       | -          | phi     | no
        g                  | Gravitational acceleration       | no       | kg m/s^2   | 9.81    | no
        Tol                | Tolerance for Colebrook eq.      | no       | -          | 1e-6    | no
        Nitr               | Max iterations for Colebrook eq. | no       | -          | 20      | no
        fDiff              | Max diff. between f and fInitial | no       | -          | 0.05    | no
    \endtable

    Example of the boundary condition specification:
    \verbatim
    <patchName>
    {
        type            headLossPressure;
        pFar            uniform 0;  // m2/s2
        HFar            10;         // m
        dP              1;          // m
        frictionLossFactors
        (
            ((1 2e-05 8) pipe1)
            ((1 2e-05 5) pipe2)
        ); // ((m, m, m) -)
        minorLossFactors
        (
            ((1 0.05) bend)
            ((1 0.01) valve)
        ); // ((m, -) -)
    }

    The input of frictionLossFactors is based on a Tuple2<vector, word> and the components
    correspond to the hydraulic diameter of the loss [m], surface roughness (epsilon) [m], 
    pipe length [m], and name of the loss e.g:
    frictionLossFactors
    ( 
        (( diameter_1 epsilon_1 L_1 ) pipe1 )
        (( diameter_2 epsilon_2 L_2 ) pipe2 )
    );

    The input of minorLossFactors is based on a Tuple2<vector2D, word> and the components
    correspond to the hydraulic diameter of the loss [m], minor loss coefficient (k), and 
    name of the loss e.g:
    minorLossFactors
    ( 
        (( diameter_1 k_1 ) valve)
        (( diameter_2 k_2 ) bend)
    );

    All parameters must be supplied in SI units, i.e. [m].

    \endverbatim

See also
    Foam::fixedValueFvPatchField

SourceFiles
    headLossPressureFvPatchScalarField.C

\*---------------------------------------------------------------------------*/

#ifndef headLossPressureFvPatchScalarField_H
#define headLossPressureFvPatchScalarField_H

#include "fixedValueFvPatchFields.H"
#include "vector2DField.H"
#include "Tuple2.H"
#include "Function1.H"


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
              Class headLossPressureFvPatchScalarField Declaration
\*---------------------------------------------------------------------------*/

class headLossPressureFvPatchScalarField
:
    public fixedValueFvPatchScalarField
{
    // Private data

        //- Name of the velocity field
        word UName_;

        //- Name of the flux transporting the field
        word phiName_;

        //- Pressure far from the patch
        scalarField pFar_;
            
        //- Hydraulic diameter of the patch
        const scalar dP_;

        //- Gravitational acceleration
        const scalar g_;

        //- Tuple2 list of the friction loss factors ((d, epsilon, L) name)
        typedef Tuple2<vector, word> indexedVector;      
        const List<indexedVector> frictionLossFactors_;

        //- Tuple2 list of the minor loss factors ((d, k) name)
        typedef Tuple2<vector2D, word> indexedVector2D;      
        const List<indexedVector2D> minorLossFactors_;

        //- Head far from patch, e.g. reservoir
        scalar HFar_;

        //- Minor loss coefficient with Function1
        autoPtr<Function1<scalar>> kDynamic_;

        //- Diameter of kDynamic_
        scalar dkDynamic_; 

        //- Flow rate to the reservoir
        autoPtr<Function1<scalar>> flowRate_;

        //- Surface area of the reservoir 
        scalar Ar_; 

        //- Previous time step
        scalar tOld_;

        //- Sum of volumetric face flux
        scalar sumPhi_;

        //- Hydralic diameter far from patch
        scalar dFar_;

        //- Scale patch velocity with UFarScale_
        scalar UFarScale_;

        //- Tolerance of the Colebrook equation iteration.
        const scalar Tol_;

        //- Maximum number of iterations for the Colebrook equation.
        const scalar Nitr_;

        //- Maximum devation between f and fInitial.
        const scalar fDiff_;

public:

    //- Runtime type information
    TypeName("headLossPressure");
    
    // Constructors

        //- Construct from patch and internal field
        headLossPressureFvPatchScalarField
        (
            const fvPatch&,
            const DimensionedField<scalar, volMesh>&
        );

        //- Construct from patch, internal field and dictionary
        headLossPressureFvPatchScalarField
        (
            const fvPatch&,
            const DimensionedField<scalar, volMesh>&,
            const dictionary&
        );

        //- Construct by mapping given headLossPressureFvPatchScalarField
        //  onto a new patch
        headLossPressureFvPatchScalarField
        (
            const headLossPressureFvPatchScalarField&,
            const fvPatch&,
            const DimensionedField<scalar, volMesh>&,
            const fvPatchFieldMapper&
        );

        //- Construct as copy
        headLossPressureFvPatchScalarField
        (
            const headLossPressureFvPatchScalarField&
        );

        //- Construct and return a clone
        virtual tmp<fvPatchScalarField> clone() const
        {
            return tmp<fvPatchScalarField>
            (
                new headLossPressureFvPatchScalarField(*this)
            );
        }

        //- Construct as copy setting internal field reference
        headLossPressureFvPatchScalarField
        (
            const headLossPressureFvPatchScalarField&,
            const DimensionedField<scalar, volMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchScalarField> clone
        (
            const DimensionedField<scalar, volMesh>& iF
        ) const
        {
            return tmp<fvPatchScalarField>
            (
                new headLossPressureFvPatchScalarField(*this, iF)
            );
        }


    // Member functions


            //- Return the name of the velocity field
            const scalar& HFar() 
            {
                return HFar_;
            }
        // Access

            //- Return the name of the velocity field
            const word& UName() const
            {
                return UName_;
            }

            //- Return reference to the name of the velocity field
            //  to allow adjustment
            word& UName()
            {
                return UName_;
            }

            //- Return the name of the flux field
            const word& phiName() const
            {
                return phiName_;
            }

            //- Return reference to the name of the flux field
            //  to allow adjustment
            word& phiName()
            {
                return phiName_;
            }

            // Return the friction loss coefficient f
            scalar fFunc
            (
                const scalar& diameterLoss,
                const scalar& surfaceRoughness,
                const scalar& UPatch,
                const scalar& nu,
                const label& frictionLossFactorsIndex
            );

            //- Return the total pressure
            const scalarField& pFar() const
            {
                return pFar_;
            }

            //- Return reference to the total pressure to allow adjustment
            scalarField& pFar()
            {
                return pFar_;
            }

            // Mapping functions

            //- Map (and resize as needed) from self given a mapping object
            virtual void autoMap
            (
                const fvPatchFieldMapper&
            );

            //- Reverse map the given fvPatchField onto this fvPatchField
            virtual void rmap
            (
                const fvPatchScalarField&,
                const labelList&
            );


        // Evaluation functions

            //- Inherit updateCoeffs from fixedValueFvPatchScalarField
            using fixedValueFvPatchScalarField::updateCoeffs;

            //- Update the coefficients associated with the patch field
            //  using the given patch total pressure and velocity fields
            virtual void updateCoeffs
            (
                const scalarField& pFarp,
                const vectorField& Up
            );
          
            //- Update the coefficients associated with the patch field
            virtual void updateCoeffs();


        //- Write
        virtual void write(Ostream&) const;
        
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
